<?php

declare(strict_types=1);

namespace Lobage\Planify\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Lobage\Planify\Traits\HasResetDate;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PlanSubscriptionUsage extends Model
{

    use HasResetDate;

    /**
     * {@inheritdoc}
     */
    protected $fillable = [
        'subscription_id',
        'feature_id',
        'used',
        'valid_until',
    ];

    /**
     * {@inheritdoc}
     */
    protected $casts = [
        'used' => 'integer',
        'valid_until' => 'datetime',
    ];

    /**
     * Create a new Eloquent model instance.
     *
     * @param array $attributes
     */
    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);

        $this->setTable(config('planify.tables.plan_subscription_usage'));
    }

    /**
     * Get validation rules
     * @return string[]
     */
    public function getRules(): array
    {
        return [
            'subscription_id' => 'required|integer|exists:' . config('planify.tables.plan_subscriptions') . ',id',
            'feature_id' => 'required|integer|exists:' . config('planify.tables.plan_features') . ',id',
            'used' => 'required|integer',
            'valid_until' => 'nullable|date',
        ];
    }

    /**
     * Subscription usage always belongs to a plan subscription feature.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function feature(): BelongsTo
    {
        return $this->belongsTo(config('planify.models.plan_feature'), 'feature_id', 'id', 'feature');
    }


    public function subscription(): BelongsTo
    {
        return $this->belongsTo(config('planify.models.plan_subscription'), 'subscription_id', 'id', 'subscription');
    }

    /**
     * Scope subscription usage by feature tag.
     *
     * @param \Illuminate\Database\Eloquent\Builder $builder
     * @param string $featureTag
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeByFeatureTag(Builder $builder, string $featureTag): Builder
    {
        return $builder->whereHas('feature', function (Builder $query) use ($featureTag) {
            $query->where('tag', $featureTag);
        });
    }

    /**
     * Check whether usage has been expired or not.
     *
     * @return bool
     */
    public function hasExpired(): bool
    {
        if (is_null($this->valid_until)) {
            return false;
        }

        return Carbon::now()->gte($this->valid_until);
    }
}
